<?php
/**
 * Plugin Name: LoginWA OTP
 * Description: OTP login/verification plugin for LoginWA with customizable login page and shortcode.
 * Version: 0.1.6
 * Author: LoginWA
 * Text Domain: loginwa-otp
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (! defined('ABSPATH')) {
    exit;
}

class LoginWA_OTP_Plugin
{
    const OPTION_KEY = 'loginwa_otp_options';

    public function __construct()
    {
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_init', [$this, 'maybe_create_login_page']);
        add_action('init', [$this, 'register_assets']);
        add_action('init', [$this, 'add_rewrite_rules']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('wp_enqueue_scripts', [$this, 'maybe_enqueue_assets']);
        add_action('init', [$this, 'register_block']);
        add_shortcode('loginwa_otp_form', [$this, 'render_shortcode']);
        add_shortcode('loginwa_otp_login', [$this, 'render_login_shortcode']);
        add_action('rest_api_init', [$this, 'register_routes']);
        add_action('login_init', [$this, 'maybe_redirect_login']);
        add_filter('login_url', [$this, 'filter_login_url'], 10, 3);
        add_action('template_redirect', [$this, 'maybe_force_site_login']);

        register_activation_hook(__FILE__, [$this, 'on_activate']);
    }

    public function on_activate(): void
    {
        $this->maybe_create_login_page(true);
        $this->add_rewrite_rules();
        flush_rewrite_rules();
    }

    public function add_rewrite_rules(): void
    {
        add_rewrite_rule('^login/?$', 'index.php?pagename=login', 'top');
    }

    public function register_admin_menu(): void
    {
        add_options_page(
            __('LoginWA OTP', 'loginwa-otp'),
            __('LoginWA OTP', 'loginwa-otp'),
            'manage_options',
            'loginwa-otp',
            [$this, 'render_settings_page']
        );
    }

    public function register_settings(): void
    {
        register_setting(self::OPTION_KEY, self::OPTION_KEY, [
            'type' => 'array',
            'sanitize_callback' => [$this, 'sanitize_options'],
            'default' => [],
        ]);

        add_settings_section('loginwa_otp_main', __('General', 'loginwa-otp'), function () {
            echo '<p>'.esc_html__('Configure LoginWA OTP defaults and login page appearance.', 'loginwa-otp').'</p>';
        }, 'loginwa-otp');

        $fields = [
            'api_key' => [__('API Key', 'loginwa-otp'), 'text'],
            'api_secret' => [__('API Secret', 'loginwa-otp'), 'text'],
            'default_country' => [__('Default country code (e.g. +62)', 'loginwa-otp'), 'text'],
            'otp_length' => [__('OTP length (digits)', 'loginwa-otp'), 'number'],
            'resend_cooldown' => [__('Resend cooldown (seconds)', 'loginwa-otp'), 'number'],
            'max_attempts' => [__('Max attempts per session', 'loginwa-otp'), 'number'],
            'enforce_login' => [__('Require OTP on WP login', 'loginwa-otp'), 'checkbox'],
            'login_redirect' => [__('Login redirect URL (after verify)', 'loginwa-otp'), 'text'],
            'login_page' => [__('Custom login page URL (optional)', 'loginwa-otp'), 'text'],
            'force_site_login' => [__('Force login for site visitors', 'loginwa-otp'), 'checkbox'],
            'hero_heading' => [__('Login heading', 'loginwa-otp'), 'text'],
            'hero_subheading' => [__('Login subheading', 'loginwa-otp'), 'text'],
            'hero_description' => [__('Login description', 'loginwa-otp'), 'textarea'],
            'badge_text' => [__('Badge text', 'loginwa-otp'), 'text'],
            'hint_text' => [__('Send button hint', 'loginwa-otp'), 'text'],
            'accent_color' => [__('Accent color (hex)', 'loginwa-otp'), 'text'],
            'card_max_width' => [__('Card max width (px)', 'loginwa-otp'), 'number'],
            'border_radius' => [__('Card radius (px)', 'loginwa-otp'), 'number'],
            'body_gap' => [__('Layout gap (px)', 'loginwa-otp'), 'number'],
            'form_padding' => [__('Card padding (px)', 'loginwa-otp'), 'number'],
            'layout' => [__('Layout (split/stacked)', 'loginwa-otp'), 'select'],
            'background_type' => [__('Background type', 'loginwa-otp'), 'select'],
            'background_from' => [__('Background color from', 'loginwa-otp'), 'text'],
            'background_to' => [__('Background color to', 'loginwa-otp'), 'text'],
            'shadow' => [__('Enable shadow', 'loginwa-otp'), 'checkbox'],
            'hero_align' => [__('Hero alignment', 'loginwa-otp'), 'select'],
            'show_hero' => [__('Show hero section', 'loginwa-otp'), 'checkbox'],
            'show_badge' => [__('Show badge', 'loginwa-otp'), 'checkbox'],
            'show_hint' => [__('Show hint', 'loginwa-otp'), 'checkbox'],
            'logo_url' => [__('Logo URL (optional)', 'loginwa-otp'), 'text'],
            'logo_alt' => [__('Logo alt text', 'loginwa-otp'), 'text'],
            'show_country' => [__('Show country dropdown', 'loginwa-otp'), 'checkbox'],
            'otp_message_template' => [__('OTP message template (use {code}, {ttl}, {app})', 'loginwa-otp'), 'textarea'],
            'send_label' => [__('Send button label', 'loginwa-otp'), 'text'],
            'verify_label' => [__('Verify button label', 'loginwa-otp'), 'text'],
            'phone_placeholder' => [__('Phone placeholder', 'loginwa-otp'), 'text'],
            'otp_placeholder' => [__('OTP placeholder', 'loginwa-otp'), 'text'],
            'theme_mode' => [__('Theme mode (light/dark/auto)', 'loginwa-otp'), 'select'],
            'show_flags' => [__('Show flags in country list', 'loginwa-otp'), 'checkbox'],
            'preferred_countries' => [__('Preferred country codes (comma)', 'loginwa-otp'), 'text'],
        ];

        foreach ($fields as $key => [$label, $type]) {
            add_settings_field(
                $key,
                esc_html($label),
                [$this, 'render_field'],
                'loginwa-otp',
                'loginwa_otp_main',
                ['key' => $key, 'type' => $type]
            );
        }
    }

    public function sanitize_options($input): array
    {
        $out = [];
        $out['api_key'] = isset($input['api_key']) ? sanitize_text_field($input['api_key']) : '';
        $out['api_secret'] = isset($input['api_secret']) ? sanitize_text_field($input['api_secret']) : '';
        $out['default_country'] = isset($input['default_country']) ? sanitize_text_field($input['default_country']) : '';
        $out['otp_length'] = isset($input['otp_length']) ? max(4, min(8, (int) $input['otp_length'])) : 6;
        $out['resend_cooldown'] = isset($input['resend_cooldown']) ? max(10, (int) $input['resend_cooldown']) : 30;
        $out['max_attempts'] = isset($input['max_attempts']) ? max(1, (int) $input['max_attempts']) : 3;
        $out['enforce_login'] = ! empty($input['enforce_login']) ? 1 : 0;
        $out['login_redirect'] = isset($input['login_redirect']) ? esc_url_raw($input['login_redirect']) : '';
        $out['login_page'] = isset($input['login_page']) ? esc_url_raw($input['login_page']) : '';
        $out['force_site_login'] = ! empty($input['force_site_login']) ? 1 : 0;
        $out['hero_heading'] = isset($input['hero_heading']) ? sanitize_text_field($input['hero_heading']) : '';
        $out['hero_subheading'] = isset($input['hero_subheading']) ? sanitize_text_field($input['hero_subheading']) : '';
        $out['hero_description'] = isset($input['hero_description']) ? sanitize_textarea_field($input['hero_description']) : '';
        $out['accent_color'] = isset($input['accent_color']) ? sanitize_hex_color($input['accent_color']) : '#0ea5e9';
        $out['card_max_width'] = isset($input['card_max_width']) ? max(280, (int) $input['card_max_width']) : 400;
        $out['border_radius'] = isset($input['border_radius']) ? max(8, (int) $input['border_radius']) : 18;
        $out['layout'] = in_array($input['layout'] ?? 'split', ['split','stacked'], true) ? $input['layout'] : 'split';
        $bgType = $input['background_type'] ?? 'default';
        $out['background_type'] = in_array($bgType, ['default','gradient','solid'], true) ? $bgType : 'default';
        $out['background_from'] = isset($input['background_from']) ? sanitize_hex_color($input['background_from']) : '#22d3ee';
        $out['background_to'] = isset($input['background_to']) ? sanitize_hex_color($input['background_to']) : '#6366f1';
        $out['shadow'] = ! empty($input['shadow']) ? 1 : 0;
        $align = $input['hero_align'] ?? 'center';
        $out['hero_align'] = in_array($align, ['left','center'], true) ? $align : 'center';
        $out['show_hero'] = array_key_exists('show_hero', $input) ? (int) ! empty($input['show_hero']) : 1;
        $out['show_badge'] = array_key_exists('show_badge', $input) ? (int) ! empty($input['show_badge']) : 1;
        $out['show_hint'] = array_key_exists('show_hint', $input) ? (int) ! empty($input['show_hint']) : 1;
        $out['body_gap'] = isset($input['body_gap']) ? max(12, (int) $input['body_gap']) : 28;
        $out['form_padding'] = isset($input['form_padding']) ? max(12, (int) $input['form_padding']) : 22;
        $out['logo_url'] = isset($input['logo_url']) ? esc_url_raw($input['logo_url']) : '';
        $out['logo_alt'] = isset($input['logo_alt']) ? sanitize_text_field($input['logo_alt']) : 'LoginWA';
        $out['badge_text'] = isset($input['badge_text']) ? sanitize_text_field($input['badge_text']) : '';
        $out['hint_text'] = isset($input['hint_text']) ? sanitize_text_field($input['hint_text']) : '';
        $out['show_country'] = ! empty($input['show_country']) ? 1 : 0;
        $out['show_flags'] = ! empty($input['show_flags']) ? 1 : 0;
        $out['preferred_countries'] = isset($input['preferred_countries']) ? sanitize_text_field($input['preferred_countries']) : '';
        $out['otp_message_template'] = isset($input['otp_message_template']) ? sanitize_textarea_field($input['otp_message_template']) : '';
        $out['send_label'] = isset($input['send_label']) ? sanitize_text_field($input['send_label']) : 'Send code';
        $out['verify_label'] = isset($input['verify_label']) ? sanitize_text_field($input['verify_label']) : 'Login';
        $out['phone_placeholder'] = isset($input['phone_placeholder']) ? sanitize_text_field($input['phone_placeholder']) : '81234567890';
        $out['otp_placeholder'] = isset($input['otp_placeholder']) ? sanitize_text_field($input['otp_placeholder']) : '123456';
        $mode = isset($input['theme_mode']) ? sanitize_text_field($input['theme_mode']) : 'auto';
        $out['theme_mode'] = in_array($mode, ['auto', 'light', 'dark'], true) ? $mode : 'auto';
        return $out;
    }

    public function render_field(array $args): void
    {
        $options = get_option(self::OPTION_KEY, []);
        $key = $args['key'];
        $type = $args['type'];
        $value = $options[$key] ?? '';
        $placeholder = $value === '' ? $this->defaultValue($key) : '';

        if ($type === 'checkbox') {
            printf(
                '<input type="checkbox" name="%1$s[%2$s]" value="1" %3$s />',
                esc_attr(self::OPTION_KEY),
                esc_attr($key),
                checked($value, 1, false)
            );
            return;
        }

        if ($type === 'select') {
            $choices = ['auto' => 'Auto (follow site)', 'light' => 'Light', 'dark' => 'Dark'];
            if ($key === 'layout') {
                $choices = ['split' => 'Split', 'stacked' => 'Stacked'];
            } elseif ($key === 'background_type') {
                $choices = ['default' => 'Default', 'gradient' => 'Custom gradient', 'solid' => 'Solid color'];
            } elseif ($key === 'hero_align') {
                $choices = ['left' => 'Left', 'center' => 'Center'];
            }
            echo '<select name="'.esc_attr(self::OPTION_KEY.'['.$key.']').'">';
            foreach ($choices as $val => $label) {
                printf('<option value="%s" %s>%s</option>', esc_attr($val), selected($value, $val, false), esc_html($label));
            }
            echo '</select>';
            return;
        }

        if ($type === 'textarea') {
            printf(
                '<textarea name="%1$s[%2$s]" rows="3" class="large-text">%3$s</textarea>',
                esc_attr(self::OPTION_KEY),
                esc_attr($key),
                esc_textarea($value)
            );
            return;
        }

        $input_type = $type === 'number' ? 'number' : 'text';
        $after = '';
        if ($key === 'api_secret') {
            $after = '<p class="description"><a href="https://loginwa.com/dashboard" target="_blank" rel="noopener">'.esc_html__('Open LoginWA Dashboard', 'loginwa-otp').'</a> '.esc_html__('to get your keys.', 'loginwa-otp').'</p>';
        }
        printf(
            '<input type="%1$s" name="%2$s[%3$s]" value="%4$s" class="regular-text" placeholder="%5$s" />%6$s',
            esc_attr($input_type),
            esc_attr(self::OPTION_KEY),
            esc_attr($key),
            esc_attr($value),
            esc_attr($placeholder),
            wp_kses_post($after)
        );
    }

    protected function defaultValue(string $key): string
    {
        $defaults = [
            'api_key' => '',
            'api_secret' => '',
            'default_country' => '62',
            'otp_length' => '6',
            'resend_cooldown' => '30',
            'max_attempts' => '3',
            'enforce_login' => '0',
            'login_redirect' => '',
            'login_page' => '',
            'force_site_login' => '0',
            'hero_heading' => 'LoginWA OTP Login',
            'hero_subheading' => '',
            'hero_description' => '',
            'badge_text' => 'Passwordless access',
            'hint_text' => 'We will text a one-time code.',
            'accent_color' => '#0ea5e9',
            'card_max_width' => '400',
            'border_radius' => '18',
            'body_gap' => '28',
            'form_padding' => '22',
            'layout' => 'split',
            'background_type' => 'default',
            'background_from' => '#22d3ee',
            'background_to' => '#6366f1',
            'shadow' => '1',
            'hero_align' => 'center',
            'show_hero' => '1',
            'show_badge' => '1',
            'show_hint' => '1',
            'logo_url' => '',
            'logo_alt' => 'LoginWA',
            'show_country' => '0',
            'show_flags' => '0',
            'preferred_countries' => '',
            'otp_message_template' => '',
            'send_label' => 'Send code',
            'verify_label' => 'Login',
            'phone_placeholder' => '81234567890',
            'otp_placeholder' => '123456',
            'theme_mode' => 'auto',
        ];
        return $defaults[$key] ?? '';
    }

    public function render_settings_page(): void
    {
        $tabs = [
            'general' => [
                'label' => __('Credentials & OTP', 'loginwa-otp'),
                'fields' => ['api_key','api_secret','default_country','otp_length','resend_cooldown','max_attempts','otp_message_template'],
            ],
            'login' => [
                'label' => __('Login Flow', 'loginwa-otp'),
                'fields' => ['enforce_login','force_site_login','login_redirect','login_page'],
            ],
            'appearance' => [
                'label' => __('Appearance & Copy', 'loginwa-otp'),
                'fields' => [
                    'hero_heading','hero_subheading','hero_description','badge_text','hint_text','hero_align',
                    'accent_color','background_type','background_from','background_to',
                    'card_max_width','border_radius','shadow','layout','show_country','show_flags','preferred_countries','theme_mode',
                    'send_label','verify_label','phone_placeholder','otp_placeholder'
                ],
            ],
            'docs' => [
                'label' => __('Docs', 'loginwa-otp'),
                'fields' => [],
            ],
            'builder' => [
                'label' => __('Builder', 'loginwa-otp'),
                'fields' => [],
            ],
        ];

        echo '<div class="wrap loginwa-admin"><h1>'.esc_html__('LoginWA OTP', 'loginwa-otp').'</h1>';
        echo '<p class="description">'.esc_html__('Configure API credentials, OTP rules, login flow, and appearance.', 'loginwa-otp').'</p>';
        echo '<div class="loginwa-shortcodes"><strong>'.esc_html__('Shortcodes', 'loginwa-otp').'</strong><br />';
        echo '<div class="loginwa-shortcodes__row"><label>[loginwa_otp_login]</label><input type="text" readonly value="[loginwa_otp_login]" onclick="this.select();" /></div>';
        echo '<div class="loginwa-shortcodes__row"><label>[loginwa_otp_form]</label><input type="text" readonly value="[loginwa_otp_form]" onclick="this.select();" /></div>';
        echo '<small>'.esc_html__('Use these shortcodes in any page or post, or drop the “LoginWA OTP Login” Gutenberg block.', 'loginwa-otp').'</small>';
        echo '</div>';
        echo '<div class="loginwa-admin__tabs">';
        foreach ($tabs as $slug => $tab) {
            echo '<button type="button" class="loginwa-admin__tab" data-target="'.esc_attr($slug).'">'.esc_html($tab['label']).'</button>';
        }
        echo '</div>';

        echo '<form method="post" action="options.php" class="loginwa-admin__form">';
        settings_fields(self::OPTION_KEY);
        foreach ($tabs as $slug => $tab) {
            echo '<div class="loginwa-admin__panel" data-panel="'.esc_attr($slug).'">';
            if (!empty($tab['fields'])) {
                echo '<table class="form-table"><tbody>';
                foreach ($tab['fields'] as $fieldKey) {
                    echo '<tr>';
                    echo '<th scope="row"><label>'.esc_html($this->fieldLabel($fieldKey)).'</label></th>';
                    echo '<td>';
                    $this->render_field(['key' => $fieldKey, 'type' => $this->fieldType($fieldKey)]);
                    echo '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
            } else {
                if ($slug === 'docs') {
                    $this->render_docs();
                } elseif ($slug === 'builder') {
                    $this->render_builder_tab();
                }
            }
            echo '</div>';
        }
        submit_button();
        echo '</form></div>';
    }

    protected function fieldLabel(string $key): string
    {
        $map = [
            'api_key' => __('API Key', 'loginwa-otp'),
            'api_secret' => __('API Secret', 'loginwa-otp'),
            'default_country' => __('Default country code (e.g. +62)', 'loginwa-otp'),
            'otp_length' => __('OTP length (digits)', 'loginwa-otp'),
            'resend_cooldown' => __('Resend cooldown (seconds)', 'loginwa-otp'),
            'max_attempts' => __('Max attempts per session', 'loginwa-otp'),
            'otp_message_template' => __('OTP message template', 'loginwa-otp'),
            'enforce_login' => __('Require OTP on WP login', 'loginwa-otp'),
            'force_site_login' => __('Force login for site visitors', 'loginwa-otp'),
            'login_redirect' => __('Login redirect URL (after verify)', 'loginwa-otp'),
            'login_page' => __('Custom login page URL (optional)', 'loginwa-otp'),
            'hero_heading' => __('Login heading', 'loginwa-otp'),
            'hero_subheading' => __('Login subheading', 'loginwa-otp'),
            'hero_description' => __('Login description', 'loginwa-otp'),
            'badge_text' => __('Badge text', 'loginwa-otp'),
            'hint_text' => __('Send button hint', 'loginwa-otp'),
            'accent_color' => __('Accent color (hex)', 'loginwa-otp'),
            'card_max_width' => __('Card max width (px)', 'loginwa-otp'),
            'border_radius' => __('Card radius (px)', 'loginwa-otp'),
            'body_gap' => __('Layout gap (px)', 'loginwa-otp'),
            'form_padding' => __('Card padding (px)', 'loginwa-otp'),
            'layout' => __('Layout (split/stacked)', 'loginwa-otp'),
            'background_type' => __('Background type', 'loginwa-otp'),
            'background_from' => __('Background color from', 'loginwa-otp'),
            'background_to' => __('Background color to', 'loginwa-otp'),
            'shadow' => __('Enable shadow', 'loginwa-otp'),
            'hero_align' => __('Hero alignment', 'loginwa-otp'),
            'show_hero' => __('Show hero section', 'loginwa-otp'),
            'show_badge' => __('Show badge', 'loginwa-otp'),
            'show_hint' => __('Show hint', 'loginwa-otp'),
            'logo_url' => __('Logo URL (optional)', 'loginwa-otp'),
            'logo_alt' => __('Logo alt text', 'loginwa-otp'),
            'show_country' => __('Show country dropdown', 'loginwa-otp'),
            'show_flags' => __('Show flags in dropdown', 'loginwa-otp'),
            'preferred_countries' => __('Preferred country codes (comma separated)', 'loginwa-otp'),
            'theme_mode' => __('Theme mode (light/dark/auto)', 'loginwa-otp'),
            'send_label' => __('Send button label', 'loginwa-otp'),
            'verify_label' => __('Verify button label', 'loginwa-otp'),
            'phone_placeholder' => __('Phone placeholder', 'loginwa-otp'),
            'otp_placeholder' => __('OTP placeholder', 'loginwa-otp'),
        ];
        return $map[$key] ?? $key;
    }

    protected function fieldType(string $key): string
    {
        $numbers = ['otp_length','resend_cooldown','max_attempts','card_max_width','border_radius','body_gap','form_padding'];
        $checkbox = ['enforce_login','force_site_login','show_country','shadow','show_flags','show_hero','show_badge','show_hint'];
        $textarea = ['hero_description','otp_message_template','hint_text'];
        if (in_array($key, $numbers, true)) return 'number';
        if (in_array($key, $checkbox, true)) return 'checkbox';
        if (in_array($key, $textarea, true)) return 'textarea';
        if (in_array($key, ['theme_mode','layout','background_type','hero_align'], true)) return 'select';
        return 'text';
    }

    protected function render_builder_tab(): void
    {
        $createUrl = admin_url('post-new.php?post_type=page');
        ?>
        <div class="loginwa-docs">
            <h2><?php echo esc_html__('Drag & drop builder (Gutenberg)', 'loginwa-otp'); ?></h2>
            <p><?php echo wp_kses_post(__('The builder is available as a block: <strong>LoginWA OTP Builder</strong>. Add this block in the Gutenberg editor, then drag-and-drop the Hero and Form blocks.', 'loginwa-otp')); ?></p>
            <ol>
                <li><a href="<?php echo esc_url($createUrl); ?>" class="button button-primary" target="_blank" rel="noopener"><?php echo esc_html__('Create a new page', 'loginwa-otp'); ?></a> <?php echo esc_html__('and open the editor.', 'loginwa-otp'); ?></li>
                <li><?php echo esc_html__('Add the “LoginWA OTP Builder” block. Inside it you can reorder the Hero and Form child blocks.', 'loginwa-otp'); ?></li>
                <li><?php echo esc_html__('Adjust the wrapper (theme, background, gap, padding) in the right panel. Configure Hero (logo, badge, heading) and Form (country dropdown, hint, labels) in their respective blocks.', 'loginwa-otp'); ?></li>
            </ol>
            <p><?php echo wp_kses_post(__('Quick option: use the single block “LoginWA OTP Login” or shortcodes <code>[loginwa_otp_login]</code>/<code>[loginwa_otp_form]</code>.', 'loginwa-otp')); ?></p>
        </div>
        <?php
    }

    protected function render_docs(): void
    {
        ?>
        <div class="loginwa-docs">
            <h2><?php echo esc_html__('How to use LoginWA OTP', 'loginwa-otp'); ?></h2>
            <ol>
                <li><?php echo wp_kses_post(__('<strong>Create API keys</strong> in your LoginWA dashboard (Apps & API Keys). Copy the public/secret keys.', 'loginwa-otp')); ?></li>
                <li><?php echo wp_kses_post(__('<strong>Configure credentials</strong> in “Credentials & OTP” tab: API Key, API Secret, default country, OTP length, cooldown, max attempts, and (optionally) custom OTP message template using {code}/{ttl}/{app}.', 'loginwa-otp')); ?></li>
                <li><?php echo wp_kses_post(__('<strong>Set login behavior</strong> in “Login Flow”: Require OTP on WP login, force site login for visitors, set redirect URL after verification, and choose custom login page (auto-created at /login by default).', 'loginwa-otp')); ?></li>
                <li><?php echo wp_kses_post(__('<strong>Design the page</strong> in “Appearance & Copy”: heading, subheading, description, badge/hint, accent color, background (default/solid/gradient), layout (split/stacked), shadow, radius, hero alignment, country dropdown options, labels, placeholders, and theme (light/dark/auto).', 'loginwa-otp')); ?></li>
                <li><?php echo wp_kses_post(__('<strong>Add to pages</strong>: use shortcodes [loginwa_otp_login] or [loginwa_otp_form], or insert the Gutenberg block “LoginWA OTP Login” and tweak its inspector settings.', 'loginwa-otp')); ?></li>
                <li><?php echo wp_kses_post(__('<strong>Country dropdown</strong>: enable “Show country dropdown,” optionally show flags, and set preferred country codes (comma) to pin them on top. Default country is used for preselect.', 'loginwa-otp')); ?></li>
                <li><?php echo wp_kses_post(__('<strong>Testing</strong>: publish the page, fill phone number, send OTP, enter code, and verify. Ensure your API key domain matches the site domain in LoginWA dashboard to avoid 401/403.', 'loginwa-otp')); ?></li>
                <li><?php echo wp_kses_post(__('<strong>Templates & copy</strong>: adjust heading/description and button labels for your brand; use the OTP message template if you want custom text in the WhatsApp message.', 'loginwa-otp')); ?></li>
            </ol>
            <p><?php echo wp_kses_post(__('<strong>Shortcodes:</strong> [loginwa_otp_login] (full login flow) and [loginwa_otp_form] (compact form). You can pass attributes like heading, description, accent_color, layout, background_type/from/to, badge, hint, show_country, show_flags, phone_placeholder, otp_placeholder, send_label, verify_label, theme_mode. Need keys? Visit <a href="https://loginwa.com/dashboard" target="_blank" rel="noopener">loginwa.com/dashboard</a>.', 'loginwa-otp')); ?></p>
        </div>
        <?php
    }

    protected function inline_background(string $type, string $from, string $to): string
    {
        if ($type === 'solid') {
            return "background: {$from};";
        }
        if ($type === 'gradient') {
            return "background: linear-gradient(135deg, {$from}, {$to});";
        }
        return '';
    }

    public function render_shortcode($atts = []): string
    {
        $defaults = [
            'heading' => '',
            'description' => '',
            'accent_color' => '',
            'card_max_width' => '',
            'show_country' => '',
            'theme_mode' => '',
            'send_label' => '',
            'verify_label' => '',
            'phone_placeholder' => '',
            'otp_placeholder' => '',
            'badge' => '',
            'hint' => '',
            'layout' => '',
            'background_type' => '',
            'background_from' => '',
            'background_to' => '',
            'border_radius' => '',
            'shadow' => '',
            'hero_align' => '',
            'show_hero' => '',
            'show_badge' => '',
            'show_hint' => '',
            'body_gap' => '',
            'form_padding' => '',
            'logo_url' => '',
            'logo_alt' => '',
        ];
        $attrs = shortcode_atts($defaults, $atts);
        $options = get_option(self::OPTION_KEY, []);
        $theme = $attrs['theme_mode'] ?: ($options['theme_mode'] ?? 'auto');
        $nonce = wp_create_nonce('loginwa-otp');
        $rest_url = esc_url_raw(rest_url('loginwa-otp/v1'));
        $accent = $attrs['accent_color'] ?: ($options['accent_color'] ?? '#0ea5e9');
        $cardMax = (int) ($attrs['card_max_width'] ?: ($options['card_max_width'] ?? 400));
        $badge = $attrs['badge'] ?: ($options['badge_text'] ?? __('Secure OTP', 'loginwa-otp'));
        $hint = $attrs['hint'] ?: ($options['hint_text'] ?? __('We will text a one-time code.', 'loginwa-otp'));
        $layout = $attrs['layout'] ?: ($options['layout'] ?? 'split');
        $bgType = $attrs['background_type'] ?: ($options['background_type'] ?? 'default');
        $bgFrom = $attrs['background_from'] ?: ($options['background_from'] ?? '#22d3ee');
        $bgTo = $attrs['background_to'] ?: ($options['background_to'] ?? '#6366f1');
        $radius = (int) ($attrs['border_radius'] ?: ($options['border_radius'] ?? 18));
        $shadow = $attrs['shadow'] !== '' ? filter_var($attrs['shadow'], FILTER_VALIDATE_BOOLEAN) : (! empty($options['shadow']));
        $heroAlign = $attrs['hero_align'] ?: ($options['hero_align'] ?? 'center');
        $showFlags = $options['show_flags'] ?? 0;
        $preferredCountries = $options['preferred_countries'] ?? '';
        $defaultCountry = $options['default_country'] ?? '';
        $showHero = $attrs['show_hero'] !== '' ? filter_var($attrs['show_hero'], FILTER_VALIDATE_BOOLEAN) : true;
        $showBadge = $attrs['show_badge'] !== '' ? filter_var($attrs['show_badge'], FILTER_VALIDATE_BOOLEAN) : true;
        $showHint = $attrs['show_hint'] !== '' ? filter_var($attrs['show_hint'], FILTER_VALIDATE_BOOLEAN) : true;
        $bodyGap = (int) ($attrs['body_gap'] ?: 28);
        $formPadding = (int) ($attrs['form_padding'] ?: 22);
        $logoUrl = $attrs['logo_url'] ?: '';
        $logoAlt = $attrs['logo_alt'] ?: 'LoginWA';
        ob_start();
        ?>
        <div class="loginwa-otp<?php echo $shadow ? '' : ' loginwa-otp--flat'; ?>" data-theme="<?php echo esc_attr($theme); ?>" data-layout="<?php echo esc_attr($layout); ?>" data-align="<?php echo esc_attr($heroAlign); ?>" style="<?php echo esc_attr($this->inline_background($bgType, $bgFrom, $bgTo)); ?> --loginwa-radius: <?php echo esc_attr($radius); ?>px;">
            <div class="loginwa-otp__body" style="gap:<?php echo esc_attr($bodyGap); ?>px;">
                <?php if ($showHero) : ?>
                <div class="loginwa-otp__hero">
                    <?php if ($logoUrl) : ?>
                        <div class="loginwa-otp__logo"><img src="<?php echo esc_url($logoUrl); ?>" alt="<?php echo esc_attr($logoAlt); ?>"></div>
                    <?php endif; ?>
                    <?php if ($showBadge) : ?><span class="loginwa-otp__badge"><?php echo esc_html($badge ?: __('Secure OTP', 'loginwa-otp')); ?></span><?php endif; ?>
                    <h2><?php echo esc_html($attrs['heading'] ?: ($options['hero_heading'] ?? __('LoginWA OTP Login', 'loginwa-otp'))); ?></h2>
                    <?php if (!empty($attrs['description'] ?: ($options['hero_description'] ?? ''))) : ?>
                        <div class="loginwa-otp__desc"><?php echo esc_html($attrs['description'] ?: ($options['hero_description'] ?? '')); ?></div>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                <form class="loginwa-otp__form loginwa-otp__form--compact" data-rest="<?php echo esc_attr($rest_url); ?>" data-nonce="<?php echo esc_attr($nonce); ?>" style="--loginwa-accent: <?php echo esc_attr($accent); ?>; max-width:<?php echo esc_attr($cardMax); ?>px; padding:<?php echo esc_attr($formPadding); ?>px;" data-hint-default="<?php echo esc_attr($hint); ?>" data-hint-success="<?php echo esc_attr($attrs['hint'] ?: __('Code sent.', 'loginwa-otp')); ?>" data-hint-sending="<?php echo esc_attr(__('Sending...', 'loginwa-otp')); ?>">
                    <div class="loginwa-otp__row loginwa-otp__row--country" data-show-flags="<?php echo esc_attr($showFlags ? '1' : '0'); ?>" data-preferred="<?php echo esc_attr($preferredCountries); ?>" data-default-country="<?php echo esc_attr($defaultCountry); ?>">
                        <label><?php echo esc_html__('Country code', 'loginwa-otp'); ?></label>
                        <select name="country_code" class="loginwa-otp__country"></select>
                    </div>
                    <div class="loginwa-otp__row">
                        <label><?php echo esc_html__('Phone number', 'loginwa-otp'); ?></label>
                        <input type="text" name="phone" placeholder="<?php echo esc_attr($attrs['phone_placeholder'] ?: ($options['phone_placeholder'] ?? '81234567890')); ?>" required>
                    </div>
                    <div class="loginwa-otp__row">
                        <button type="button" class="loginwa-otp__send"><?php echo esc_html($attrs['send_label'] ?: ($options['send_label'] ?? __('Send code', 'loginwa-otp'))); ?></button>
                    </div>
                    <div class="loginwa-otp__row loginwa-otp__otp-section">
                        <label><?php echo esc_html__('OTP code', 'loginwa-otp'); ?></label>
                        <input type="text" name="otp_code" placeholder="<?php echo esc_attr($attrs['otp_placeholder'] ?: ($options['otp_placeholder'] ?? '123456')); ?>" inputmode="numeric">
                    </div>
                    <div class="loginwa-otp__row loginwa-otp__otp-section">
                        <button type="button" class="loginwa-otp__verify"><?php echo esc_html($attrs['verify_label'] ?: ($options['verify_label'] ?? __('Verify', 'loginwa-otp'))); ?></button>
                    </div>
                    <div class="loginwa-otp__status" aria-live="polite"></div>
                </form>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function render_login_shortcode($atts = []): string
    {
        $defaults = [
            'heading' => '',
            'subheading' => '',
            'description' => '',
            'accent_color' => '',
            'card_max_width' => '',
            'show_country' => '',
            'theme_mode' => '',
            'send_label' => '',
            'verify_label' => '',
            'phone_placeholder' => '',
            'otp_placeholder' => '',
            'badge' => '',
            'hint' => '',
            'layout' => '',
            'background_type' => '',
            'background_from' => '',
            'background_to' => '',
            'border_radius' => '',
            'shadow' => '',
            'hero_align' => '',
            'show_hero' => '',
            'show_badge' => '',
            'show_hint' => '',
            'body_gap' => '',
            'form_padding' => '',
            'logo_url' => '',
            'logo_alt' => '',
        ];
        $attrs = shortcode_atts($defaults, $atts);
        $options = get_option(self::OPTION_KEY, []);
        $theme = $attrs['theme_mode'] ?: ($options['theme_mode'] ?? 'auto');
        $nonce = wp_create_nonce('loginwa-otp');
        $rest_url = esc_url_raw(rest_url('loginwa-otp/v1'));
        $redirect = ! empty($options['login_redirect']) ? esc_url($options['login_redirect']) : home_url();
        $accent = $attrs['accent_color'] ?: ($options['accent_color'] ?? '#0ea5e9');
        $cardMax = (int) ($attrs['card_max_width'] ?: ($options['card_max_width'] ?? 400));
        $heading = $attrs['heading'] ?: ($options['hero_heading'] ?? __('LoginWA OTP Login', 'loginwa-otp'));
        $sub = $attrs['subheading'] ?: ($options['hero_subheading'] ?? '');
        $desc = $attrs['description'] ?: ($options['hero_description'] ?? '');
        $showCountry = $attrs['show_country'] !== '' ? filter_var($attrs['show_country'], FILTER_VALIDATE_BOOLEAN) : (! empty($options['show_country']));
        $sendLabel = $attrs['send_label'] ?: ($options['send_label'] ?? __('Send code', 'loginwa-otp'));
        $verifyLabel = $attrs['verify_label'] ?: ($options['verify_label'] ?? __('Login', 'loginwa-otp'));
        $phonePh = $attrs['phone_placeholder'] ?: ($options['phone_placeholder'] ?? '81234567890');
        $otpPh = $attrs['otp_placeholder'] ?: ($options['otp_placeholder'] ?? '123456');
        $badge = $attrs['badge'] ?: ($options['badge_text'] ?? __('Passwordless access', 'loginwa-otp'));
        $hint = $attrs['hint'] ?: ($options['hint_text'] ?? __('We will text a one-time code.', 'loginwa-otp'));
        $layout = $attrs['layout'] ?: ($options['layout'] ?? 'split');
        $bgType = $attrs['background_type'] ?: ($options['background_type'] ?? 'default');
        $bgFrom = $attrs['background_from'] ?: ($options['background_from'] ?? '#22d3ee');
        $bgTo = $attrs['background_to'] ?: ($options['background_to'] ?? '#6366f1');
        $radius = (int) ($attrs['border_radius'] ?: ($options['border_radius'] ?? 18));
        $shadow = $attrs['shadow'] !== '' ? filter_var($attrs['shadow'], FILTER_VALIDATE_BOOLEAN) : (! empty($options['shadow']));
        $heroAlign = $attrs['hero_align'] ?: ($options['hero_align'] ?? 'center');
        $showFlags = $options['show_flags'] ?? 0;
        $preferredCountries = $options['preferred_countries'] ?? '';
        $defaultCountry = $options['default_country'] ?? '';
        $showHero = $attrs['show_hero'] !== '' ? filter_var($attrs['show_hero'], FILTER_VALIDATE_BOOLEAN) : true;
        $showBadge = $attrs['show_badge'] !== '' ? filter_var($attrs['show_badge'], FILTER_VALIDATE_BOOLEAN) : true;
        $showHint = $attrs['show_hint'] !== '' ? filter_var($attrs['show_hint'], FILTER_VALIDATE_BOOLEAN) : true;
        $bodyGap = (int) ($attrs['body_gap'] ?: 28);
        $formPadding = (int) ($attrs['form_padding'] ?: 22);
        $logoUrl = $attrs['logo_url'] ?: '';
        $logoAlt = $attrs['logo_alt'] ?: 'LoginWA';
        ob_start();
        ?>
        <div class="loginwa-otp<?php echo $shadow ? '' : ' loginwa-otp--flat'; ?>" data-theme="<?php echo esc_attr($theme); ?>" data-layout="<?php echo esc_attr($layout); ?>" data-align="<?php echo esc_attr($heroAlign); ?>" style="<?php echo esc_attr($this->inline_background($bgType, $bgFrom, $bgTo)); ?> --loginwa-radius: <?php echo esc_attr($radius); ?>px;">
            <div class="loginwa-otp__body" style="gap:<?php echo esc_attr($bodyGap); ?>px;">
                <?php if ($showHero) : ?>
                <div class="loginwa-otp__hero">
                    <?php if ($logoUrl) : ?>
                        <div class="loginwa-otp__logo"><img src="<?php echo esc_url($logoUrl); ?>" alt="<?php echo esc_attr($logoAlt); ?>"></div>
                    <?php endif; ?>
                    <?php if ($showBadge) : ?><span class="loginwa-otp__badge"><?php echo esc_html($badge ?: __('Passwordless access', 'loginwa-otp')); ?></span><?php endif; ?>
                    <h2><?php echo esc_html($heading); ?></h2>
                    <?php if ($sub) : ?><div class="loginwa-otp__sub"><?php echo esc_html($sub); ?></div><?php endif; ?>
                    <?php if ($desc) : ?><div class="loginwa-otp__desc"><?php echo esc_html($desc); ?></div><?php endif; ?>
                </div>
                <?php endif; ?>
                <form class="loginwa-otp__form loginwa-otp__form--login loginwa-otp__form--compact" data-rest="<?php echo esc_attr($rest_url); ?>" data-nonce="<?php echo esc_attr($nonce); ?>" data-redirect="<?php echo esc_attr($redirect); ?>" style="--loginwa-accent: <?php echo esc_attr($accent); ?>; max-width:<?php echo esc_attr($cardMax); ?>px; padding:<?php echo esc_attr($formPadding); ?>px;" data-hint-default="<?php echo esc_attr($hint); ?>" data-hint-success="<?php echo esc_attr($attrs['hint'] ?: __('Code sent.', 'loginwa-otp')); ?>" data-hint-sending="<?php echo esc_attr(__('Sending...', 'loginwa-otp')); ?>">
                    <?php if ($showCountry) : ?>
                    <div class="loginwa-otp__row loginwa-otp__row--country" data-show-flags="<?php echo esc_attr($showFlags ? '1' : '0'); ?>" data-preferred="<?php echo esc_attr($preferredCountries); ?>" data-default-country="<?php echo esc_attr($defaultCountry); ?>">
                        <label><?php echo esc_html__('Country code', 'loginwa-otp'); ?></label>
                        <select name="country_code" class="loginwa-otp__country"></select>
                    </div>
                    <?php endif; ?>
                    <div class="loginwa-otp__row">
                        <label><?php echo esc_html__('Phone number', 'loginwa-otp'); ?></label>
                        <input type="text" name="phone" placeholder="<?php echo esc_attr($phonePh); ?>" required>
                    </div>
                    <div class="loginwa-otp__row loginwa-otp__row--inline">
                        <button type="button" class="loginwa-otp__send"><?php echo esc_html($sendLabel); ?></button>
                        <?php if ($showHint) : ?><span class="loginwa-otp__hint"><?php echo esc_html($hint); ?></span><?php endif; ?>
                    </div>
                    <div class="loginwa-otp__row loginwa-otp__otp-section">
                        <label><?php echo esc_html__('OTP code', 'loginwa-otp'); ?></label>
                        <input type="text" name="otp_code" placeholder="<?php echo esc_attr($otpPh); ?>" inputmode="numeric">
                    </div>
                    <div class="loginwa-otp__row loginwa-otp__otp-section">
                        <button type="button" class="loginwa-otp__verify loginwa-otp__verify--login"><?php echo esc_html($verifyLabel); ?></button>
                    </div>
                    <div class="loginwa-otp__status" aria-live="polite"></div>
                </form>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function register_block(): void
    {
        if (! function_exists('register_block_type')) {
            return;
        }

        wp_register_script(
            'loginwa-otp-block',
            plugins_url('assets/loginwa-otp-block.js', __FILE__),
            ['wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n', 'wp-block-editor'],
            '0.1.0',
            true
        );

        if (function_exists('wp_set_script_translations')) {
            wp_set_script_translations('loginwa-otp-block', 'loginwa-otp');
        }

        register_block_type('loginwa/otp-login', [
            'editor_script' => 'loginwa-otp-block',
            'render_callback' => function ($attrs = []) {
                $shortcode = '[loginwa_otp_login';
                foreach ($attrs as $key => $val) {
                    if ($val === '' || $val === null) {
                        continue;
                    }
                    if (is_bool($val)) {
                        $val = $val ? 'true' : 'false';
                    }
                    $shortcode .= ' '.sanitize_key($key).'="'.esc_attr($val).'"';
                }
                $shortcode .= ']';
                return do_shortcode($shortcode);
            },
            'attributes' => [
                'heading' => ['type' => 'string', 'default' => 'LoginWA OTP Login'],
                'subheading' => ['type' => 'string', 'default' => ''],
                'description' => ['type' => 'string', 'default' => ''],
                'accent_color' => ['type' => 'string', 'default' => '#22d3ee'],
                'card_max_width' => ['type' => 'number', 'default' => 420],
                'show_country' => ['type' => 'boolean', 'default' => true],
                'theme_mode' => ['type' => 'string', 'default' => 'auto'],
                'send_label' => ['type' => 'string', 'default' => 'Send code'],
                'verify_label' => ['type' => 'string', 'default' => 'Login'],
                'phone_placeholder' => ['type' => 'string', 'default' => '81234567890'],
                'otp_placeholder' => ['type' => 'string', 'default' => '123456'],
                'badge' => ['type' => 'string', 'default' => 'Passwordless access'],
                'hint' => ['type' => 'string', 'default' => 'We will text a one-time code.'],
                'layout' => ['type' => 'string', 'default' => 'split'],
                'background_type' => ['type' => 'string', 'default' => 'default'],
                'background_from' => ['type' => 'string', 'default' => '#22d3ee'],
                'background_to' => ['type' => 'string', 'default' => '#6366f1'],
                'border_radius' => ['type' => 'number', 'default' => 18],
                'shadow' => ['type' => 'boolean', 'default' => true],
                'hero_align' => ['type' => 'string', 'default' => 'left'],
                'show_hero' => ['type' => 'boolean', 'default' => true],
                'show_badge' => ['type' => 'boolean', 'default' => true],
                'show_hint' => ['type' => 'boolean', 'default' => true],
                'body_gap' => ['type' => 'number', 'default' => 28],
                'form_padding' => ['type' => 'number', 'default' => 22],
                'logo_url' => ['type' => 'string', 'default' => ''],
                'logo_alt' => ['type' => 'string', 'default' => 'LoginWA'],
            ],
        ]);
    }

    protected function get_login_page_url(): string
    {
        $options = get_option(self::OPTION_KEY, []);
        $url = trim((string) ($options['login_page'] ?? ''));
        if ($url) {
            return $url;
        }
        $existing = get_page_by_path('login');
        return $existing ? get_permalink($existing->ID) : '';
    }

    protected function get_login_page_id(): int
    {
        $url = $this->get_login_page_url();
        if ($url) {
            $id = url_to_postid($url);
            if ($id) {
                return (int) $id;
            }
        }
        $existing = get_page_by_path('login');
        return $existing ? (int) $existing->ID : 0;
    }

    public function maybe_create_login_page(bool $force = false): void
    {
        $options = get_option(self::OPTION_KEY, []);
        $url = trim((string) ($options['login_page'] ?? ''));
        $desired_slug = 'login';

        // If admin set a custom URL different from /login, honor it.
        if (! $force && $url && strpos(untrailingslashit($url), '/'.$desired_slug) === false) {
            return;
        }

        $existing = get_page_by_path($desired_slug);
        if (! $existing) {
            $pages = get_posts([
                'post_type' => 'page',
                'title' => 'LoginWA OTP Login',
                'posts_per_page' => 1,
                'post_status' => 'any',
                'fields' => 'all',
            ]);
            $existing = ! empty($pages) ? $pages[0] : null;
        }

        if ($existing) {
            $permalink = get_permalink($existing->ID);
            $options['login_page'] = $permalink;
            update_option(self::OPTION_KEY, $options);
            return;
        }

        $page_id = wp_insert_post([
            'post_title' => 'LoginWA OTP Login',
            'post_content' => '[loginwa_otp_login]',
            'post_status' => 'publish',
            'post_type' => 'page',
            'post_name' => $desired_slug,
        ]);

        if ($page_id && ! is_wp_error($page_id)) {
            $permalink = get_permalink($page_id);
            $options['login_page'] = $permalink;
            update_option(self::OPTION_KEY, $options);
        }
    }

    public function maybe_redirect_login(): void
    {
        $custom = $this->get_login_page_url();
        if (empty($custom)) {
            return;
        }

        $action = filter_input(INPUT_GET, 'action', FILTER_SANITIZE_SPECIAL_CHARS);
        if ($action === 'logout') {
            return;
        }
        if (defined('REST_REQUEST') && REST_REQUEST) {
            return;
        }
        $raw_request_uri = isset($_SERVER['REQUEST_URI']) ? esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($raw_request_uri && str_contains($raw_request_uri, '/wp-json/')) {
            return;
        }
        if (wp_doing_ajax()) {
            return;
        }

        $customUrl = untrailingslashit($custom);
        $currentPath = wp_parse_url($raw_request_uri, PHP_URL_PATH) ?: '';
        $current = untrailingslashit(home_url($currentPath));

        if ($current !== $customUrl) {
            wp_safe_redirect($customUrl);
            exit;
        }
    }

    public function filter_login_url($login_url, $redirect, $force_reauth)
    {
        $custom = $this->get_login_page_url();
        return $custom ? $custom : $login_url;
    }

    public function maybe_force_site_login(): void
    {
        $options = get_option(self::OPTION_KEY, []);
        if (empty($options['force_site_login'])) {
            return;
        }

        if (is_user_logged_in()) {
            return;
        }

        if (is_admin() || wp_doing_ajax() || (defined('REST_REQUEST') && REST_REQUEST)) {
            return;
        }
        $raw_request_uri = isset($_SERVER['REQUEST_URI']) ? esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($raw_request_uri && str_contains($raw_request_uri, '/wp-json/')) {
            return;
        }

        $custom = $this->get_login_page_url();
        if (! $custom) {
            return;
        }

        $current_request = isset($GLOBALS['wp']->request) ? sanitize_text_field($GLOBALS['wp']->request) : '';
        $current = home_url(add_query_arg([], $current_request));
        if (untrailingslashit($current) === untrailingslashit($custom)) {
            return;
        }

        if (is_feed() || is_trackback() || is_preview()) {
            return;
        }

        $loginId = $this->get_login_page_id();
        if ($loginId && is_page($loginId)) {
            return;
        }

        $target = add_query_arg(['redirect_to' => rawurlencode($current)], $custom);
        wp_safe_redirect($target);
        exit;
    }

    public function register_assets(): void
    {
        $url = plugin_dir_url(__FILE__);
        wp_register_style('loginwa-otp', $url.'assets/loginwa-otp.css', [], '0.1.6');
        wp_register_style('loginwa-otp-admin', false, [], '0.1.6');
        wp_register_script('loginwa-otp-admin', '', [], '0.1.6', true);
        wp_register_script('loginwa-otp-countries', $url.'assets/loginwa-otp-countries.js', [], '0.1.6', true);
        wp_register_script('loginwa-otp', $url.'assets/loginwa-otp.js', ['loginwa-otp-countries'], '0.1.6', true);
    }

    public function enqueue_admin_assets(string $hook): void
    {
        if ($hook !== 'settings_page_loginwa-otp') {
            return;
        }

        wp_enqueue_style('loginwa-otp-admin');
        wp_add_inline_style('loginwa-otp-admin', '.loginwa-admin__tabs{display:flex;gap:8px;margin:16px 0;}.loginwa-admin__tab{padding:8px 14px;border:1px solid #ccd1dc;border-radius:6px;background:#f6f7fb;cursor:pointer;font-weight:600;}.loginwa-admin__tab.is-active{background:#2563eb;color:#fff;border-color:#1d4ed8;}.loginwa-admin__panel{display:none;margin-top:10px;}.loginwa-admin__panel.is-active{display:block;}.loginwa-shortcodes{margin-top:12px;margin-bottom:12px;padding:10px;border:1px solid #d8dce5;border-radius:6px;background:#f8fafc;}.loginwa-shortcodes__row{display:flex;gap:8px;align-items:center;margin-top:6px;}.loginwa-shortcodes__row input{width:260px;}');

        wp_enqueue_script('loginwa-otp-admin');
        wp_add_inline_script('loginwa-otp-admin', "document.addEventListener('DOMContentLoaded',function(){const tabs=[...document.querySelectorAll('.loginwa-admin__tab')];const panels=[...document.querySelectorAll('.loginwa-admin__panel')];function activate(slug){tabs.forEach(t=>t.classList.toggle('is-active',t.dataset.target===slug));panels.forEach(p=>p.classList.toggle('is-active',p.dataset.panel===slug));}tabs.forEach(t=>t.addEventListener('click',()=>activate(t.dataset.target)));if(tabs.length){activate(tabs[0].dataset.target);}});");
    }

    public function maybe_enqueue_assets(): void
    {
        if (! is_singular()) {
            return;
        }
        global $post;
        $content = $post->post_content ?? '';
        $hasShortcode = has_shortcode($content, 'loginwa_otp_form') || has_shortcode($content, 'loginwa_otp_login');
        $hasBlock = function_exists('has_block') && (has_block('loginwa/otp-login', $post) || has_block('loginwa/otp-builder', $post));
        if (! $hasShortcode && ! $hasBlock) {
            return;
        }

        $options = get_option(self::OPTION_KEY, []);
        wp_enqueue_style('loginwa-otp');
        wp_enqueue_script('loginwa-otp');
        wp_localize_script('loginwa-otp', 'LoginWAOTP', [
            'rest' => esc_url_raw(rest_url('loginwa-otp/v1')),
            'nonce' => wp_create_nonce('loginwa-otp'),
            'otpLength' => (int) ($options['otp_length'] ?? 6),
            'resendCooldown' => (int) ($options['resend_cooldown'] ?? 30),
            'maxAttempts' => (int) ($options['max_attempts'] ?? 3),
            'i18n' => [
                'phoneRequired' => __('Phone is required.', 'loginwa-otp'),
                'otpRequired' => __('OTP is required.', 'loginwa-otp'),
                'sending' => __('Sending...', 'loginwa-otp'),
                'sent' => __('OTP sent.', 'loginwa-otp'),
                'sendFailed' => __('Failed to send OTP', 'loginwa-otp'),
                'verifyFailed' => __('Verification failed', 'loginwa-otp'),
                'verified' => __('OTP verified.', 'loginwa-otp'),
            ],
        ]);
    }

    public function verify_rest_permission(\WP_REST_Request $request)
    {
        $nonce = $request->get_header('X-WP-Nonce') ?: $request->get_param('_wpnonce');
        if ($nonce && wp_verify_nonce($nonce, 'loginwa-otp')) {
            return true;
        }

        return new \WP_Error(
            'rest_forbidden',
            __('Invalid or missing security token.', 'loginwa-otp'),
            ['status' => 403]
        );
    }

    public function register_routes(): void
    {
        register_rest_route('loginwa-otp/v1', '/send', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_send'],
            'permission_callback' => [$this, 'verify_rest_permission'],
        ]);

        register_rest_route('loginwa-otp/v1', '/verify', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_verify'],
            'permission_callback' => [$this, 'verify_rest_permission'],
        ]);
    }

    public function handle_send(\WP_REST_Request $request): \WP_REST_Response
    {
        $phone = sanitize_text_field($request->get_param('phone'));
        $country = sanitize_text_field($request->get_param('country_code'));
        $options = get_option(self::OPTION_KEY, []);

        if (! $phone) {
            return new \WP_REST_Response(['success' => false, 'message' => __('Phone is required.', 'loginwa-otp')], 400);
        }

        $cc = ltrim($country, '+');
        $fullPhone = $phone;
        if ($cc && strpos($phone, '+') !== 0) {
            $fullPhone = '+'.$cc.$phone;
        }

        $endpoint = 'https://api.loginwa.com/api/v1/auth/start';
        $payload = [
            'phone' => $fullPhone,
            'country_code' => $country,
            'otp_length' => (int) ($options['otp_length'] ?? 6),
            'message_template' => !empty($options['otp_message_template']) ? $options['otp_message_template'] : null,
        ];

        $resp = wp_remote_post($endpoint, [
            'headers' => [
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                // LoginWA API expects the secret key as bearer token; public key is not required here.
                'Authorization' => 'Bearer '.($options['api_secret'] ?? ''),
            ],
            'timeout' => 20,
            'body' => wp_json_encode($payload),
        ]);

        if (is_wp_error($resp)) {
            /* translators: %s: error message from wp_remote_post */
            $error_msg = sprintf(__('Request failed: %s', 'loginwa-otp'), $resp->get_error_message());
            return new \WP_REST_Response(['success' => false, 'message' => $error_msg], 500);
        }

        $code = wp_remote_retrieve_response_code($resp);
        $raw = wp_remote_retrieve_body($resp);
        $body = json_decode($raw, true);

        if ($code >= 400 || empty($body['session_id'])) {
            $msg = $body['message'] ?? $raw ?? __('Failed to send OTP.', 'loginwa-otp');
            return new \WP_REST_Response(['success' => false, 'message' => $msg], $code ?: 400);
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => $body['message'] ?? __('OTP sent.', 'loginwa-otp'),
            'session_id' => $body['session_id'],
        ]);
    }

    public function handle_verify(\WP_REST_Request $request): \WP_REST_Response
    {
        $otp = sanitize_text_field($request->get_param('otp_code'));
        $sessionId = sanitize_text_field($request->get_param('session_id'));
        $options = get_option(self::OPTION_KEY, []);

        if (! $otp || ! $sessionId) {
            return new \WP_REST_Response(['success' => false, 'message' => __('OTP and session are required.', 'loginwa-otp')], 400);
        }

        $endpoint = 'https://api.loginwa.com/api/v1/auth/verify';
        $resp = wp_remote_post($endpoint, [
            'headers' => [
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer '.($options['api_secret'] ?? ''),
            ],
            'timeout' => 20,
            'body' => wp_json_encode([
                'session_id' => $sessionId,
                'otp_code' => $otp,
            ]),
        ]);

        if (is_wp_error($resp)) {
            /* translators: %s: error message from wp_remote_post */
            $error_msg = sprintf(__('Request failed: %s', 'loginwa-otp'), $resp->get_error_message());
            return new \WP_REST_Response(['success' => false, 'message' => $error_msg], 500);
        }

        $code = wp_remote_retrieve_response_code($resp);
        $raw = wp_remote_retrieve_body($resp);
        $body = json_decode($raw, true);

        if ($code >= 400) {
            $msg = $body['message'] ?? $raw ?? __('Verification failed.', 'loginwa-otp');
            return new \WP_REST_Response(['success' => false, 'message' => $msg], $code ?: 400);
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => $body['message'] ?? __('OTP verified.', 'loginwa-otp'),
            'user' => $body['user'] ?? null,
        ]);
    }
}

new LoginWA_OTP_Plugin();
